<?php
namespace App\Http\Controllers;

use App\Models\Event;
use App\Models\Ticket;
use App\Models\ComplimentaryCode;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\DB;
use App\Models\ComplimentaryTicket;

class ComplimentaryTicketController extends Controller
{
    protected $templates = [
        'simple' => [
            'name' => 'Simple Invitation',
            'template' => "Dear {name},\n\nWelcome to {event_name}!\n\nTicket #: {code}\nTickets:{quantity}\n\nGet your tickets here: {redeem_url}"
        ],
        'detailed' => [
            'name' => 'Detailed Invitation',
            'template' => "Dear {name},\n\nYou've received {quantity} complimentary tickets for {event_name}.\n\nCode: {code}\nQuantity: {quantity}\nRedeem here: {redeem_url}\n\nSee you there!"
        ],
    ];

    protected $emailTemplates = [
        'simple' => [
            'subject' => 'Your Complimentary Ticket for {event_name}',
            'template' => "Dear {name},\n\nWelcome to {event_name}!\n\nTicket#: {code}\nTickets: {quantity}\n\nGet your tickets here: {redeem_url}\n\n"
        ],
        'detailed' => [
            'subject' => 'Complimentary Tickets for {event_name}',
            'template' => "Dear {name},\n\nYou've received complimentary tickets for {event_name}.\n\nTicket #: {code}\nTickets: {quantity}\n\nGet your tickets here: {redeem_url}\nDownload ticket: {download_url}\n\nWe look forward to seeing you at the event!"
        ],
    ];

    // Generate codes form
    public function createCodes()
    {
        $events = Event::all();
        $highlightedEvent = Event::find(1);
        return view('complimentary.create-codes', compact('events', 'highlightedEvent'))
            ->with('templates', $this->templates);
    }

    public function storeCodes(Request $request)
    {
        $validated = $this->validateRequest($request);

        DB::beginTransaction();
        try {
            $event = Event::findOrFail($validated['event_id']);
            $code = 'COMP-' . strtoupper(Str::random(6));

            $complimentaryCode = ComplimentaryCode::create([
                'event_id' => $event->id,
                'code' => $code,
                'expires_at' => '2025-11-29 23:59:59',
                'generated_by_name' => $validated['generated_by'] ?? 'admin',
                'recipient_name' => $validated['recipient_name'],
                'recipient_phone' => $validated['recipient_phone'],
                'recipient_email' => $validated['recipient_email'],
                'ticket_quantity' => $validated['quantity'],
                'ticket_type' => $validated['ticket_type'],
                'sms_status' => 'pending',
                'email_status' => 'pending',
            ]);

            $event = $complimentaryCode->event;
            $reference = 'COMP-' . Str::uuid();

            $complimentaryTicket = ComplimentaryTicket::create([
                'event_id' => $event->id,
                'first_name' => $complimentaryCode->recipient_name,
                'last_name' => $complimentaryCode->recipient_name,
                'phone' => $complimentaryCode->recipient_phone,
                'email' => $complimentaryCode->recipient_email,
                'school' => 'school',
                'quantity' => $validated['quantity'],
                'reference' => $reference,
                'complimentary_code' => $complimentaryCode->code,
                'ticket_type' => $complimentaryCode->ticket_type ?? 'normal',
                'is_scanned' => false,
                'scanned_at' => null,
                'scanned_by' => null,
            ]);

            // Send notifications
            $message = "Dear {$complimentaryTicket->first_name} ,Welcome to {$event->name}  \nComplimentary Ticket #: {$complimentaryTicket->complimentary_code}  \nTickets: {$complimentaryTicket->quantity}  \nGet your tickets here:" . url("/get/complimentary-ticket/{$complimentaryTicket->reference}");

            $smsResult = $this->sendSmsViaWasiliana($complimentaryTicket->phone, $message);
            $emailResult = $this->sendComplimentaryEmail($complimentaryTicket->email, 'Your PACE Complimentary TeenzFest Ticket Confirmation', $complimentaryTicket, $event);

            // $emailResult = $this->sendCodeViaEmail($complimentaryCode, $validated['sms_template'], $event);

            // Update statuses
            $complimentaryCode->sms_status = $smsResult['success'] ? 'sent' : 'failed';
            $complimentaryCode->sms_error = $smsResult['success'] ? null : ($smsResult['error'] ?? 'Failed to send SMS');
            $complimentaryCode->email_status = $emailResult['success'] ? 'sent' : 'failed';
            $complimentaryCode->email_error = $emailResult['success'] ? null : ($emailResult['error'] ?? 'Failed to send email');
            $complimentaryCode->save();

            DB::commit();

            // Return response with proper status
            if ($smsResult['success'] && $emailResult['success']) {
                return redirect()->back()
                    ->with('success', 'Ticket generated and notifications sent successfully!')
                    ->with('generated_codes', [$code]);
            } else {
                $errors = [];
                if (!$smsResult['success'])
                    $errors[] = 'SMS failed: ' . $smsResult['error'];
                if (!$emailResult['success'])
                    $errors[] = 'Email failed: ' . $emailResult['error'];

                return redirect()->back()
                    ->with('warning', 'Ticket generated but some notifications failed')
                    ->with('generated_codes', [$code])
                    ->withErrors($errors);
            }

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error generating codes: ' . $e->getMessage());
            return back()->with('error', 'Error generating ticket: ' . $e->getMessage());
        }
    }
    // List all generated codes
    public function index(Request $request)
    {
        $query = ComplimentaryCode::with(['event'])->orderBy('created_at', 'desc');

        // Handle search query
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('code', 'like', "%{$search}%")
                    ->orWhere('recipient_name', 'like', "%{$search}%")
                    ->orWhere('recipient_phone', 'like', "%{$search}%")
                    ->orWhere('recipient_email', 'like', "%{$search}%");
            });
        }

        // Handle status filter
        if ($status = $request->input('status')) {
            $query->where(function ($q) use ($status) {
                $q->where('sms_status', $status)
                    ->orWhere('email_status', $status);
            });
        }

        // Handle pagination size
        $perPage = $request->input('per_page', 30); // Default to 30 if not specified
        $perPage = in_array($perPage, [25, 50, 100, 200]) ? $perPage : 30; // Restrict to valid options

        // Paginate results
        $codes = $query->paginate($perPage)->withQueryString();

        return view('complimentary.index', compact('codes'));
    }
    // Redeem code form
    public function redeemForm(Request $request)
    {
        $highlightedEvent = Event::find(1);
        $prefilledCode = $request->query('code', '');

        return view('complimentary.redeem', [
            'highlightedEvent' => $highlightedEvent,
            'prefilledCode' => $prefilledCode
        ]);
    }

    public function redeem(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string',
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
        ]);

        $code = strtoupper(trim($validated['code']));
        $complimentaryCode = ComplimentaryCode::whereRaw('UPPER(code) = ?', [$code])
            ->where('expires_at', '>', now())
            ->whereColumn('redeemed_count', '<', 'ticket_quantity')
            ->first();

        if (!$complimentaryCode) {
            return $request->ajax()
                ? response()->json(['success' => false, 'message' => 'Invalid, expired, or fully redeemed code.'], 422)
                : back()->withInput()->with('error', 'Invalid, expired, or fully redeemed code.');
        }

        $event = $complimentaryCode->event;
        $reference = 'COMP-' . Str::uuid();

        $ticket = ComplimentaryTicket::create([
            'event_id' => $event->id,
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'phone' => $validated['phone'],
            'email' => $validated['email'],
            'school' => 'school',
            'quantity' => 1,
            'reference' => $reference,
            'complimentary_code' => $complimentaryCode->code,
            'ticket_type' => $complimentaryCode->ticket_type ?? 'normal',
            'is_scanned' => false,
            'scanned_at' => null,
            'scanned_by' => null,
        ]);

        $complimentaryCode->increment('redeemed_count');
        if ($complimentaryCode->redeemed_count >= $complimentaryCode->ticket_quantity) {
            $complimentaryCode->update([
                'is_used' => true,
                'used_at' => now(),
                'used_by_name' => $validated['email'],
            ]);
        }

        return $request->ajax()
            ? response()->json([
                'success' => true,
                'message' => 'Complimentary ticket created successfully!',
                'redirect' => route('complimentary.ticket.preview', $ticket->id)
            ])
            : redirect()->route('complimentary.ticket.preview', $ticket->id)
                ->with('success', 'Complimentary ticket created successfully!');
    }



    public function resendNotification($id)
    {
        try {
            $complimentaryCode = ComplimentaryCode::findOrFail($id);
            $event = $complimentaryCode->event;

            $complimentary = ComplimentaryTicket::where('complimentary_code', $complimentaryCode->code)->first();

            // Validate phone number first
            $phone = $this->cleanPhoneNumber($complimentary->phone);
            if (!$phone) {
                throw new \Exception('Invalid phone number format');
            }
            $message = "Dear {$complimentary->first_name} {$complimentary->last_name},Welcome to {$event->name}  \nComplimentary Ticket #: {$complimentary->complimentary_code}  \nTickets: {$complimentary->quantity}  \nGet your tickets here:" . url("/get/complimentary-ticket/{$complimentary->reference}");
            $smsResult = $this->sendSmsViaWasiliana($phone, $message);

            // Send Email
            $emailResult = $this->sendComplimentaryEmail(
                $complimentary->email,
                'Your PACE Complimentary TeenzFest Ticket Confirmation',
                $complimentary,
                $event
            );
            if ($emailResult) {
                // Update statuses
                $complimentaryCode->sms_status = 'sent';
                $complimentaryCode->sms_error = null;
                $complimentaryCode->email_status = 'sent';
                $complimentaryCode->email_error = null;
                $complimentaryCode->save();
            }

            if ($smsResult && $emailResult) {
                return back()->with('success', 'Notifications resent successfully!');
            } else {
                $errors = [];
                if (!$smsResult)
                    $errors[] = 'SMS failed: ' . ($smsResult ?? 'Unknown error');
                if (!$emailResult)
                    $errors[] = 'Email failed';

                return back()->with('warning', 'Some notifications failed to send: ' . implode(', ', $errors));
            }

        } catch (\Exception $e) {
            Log::error('Error resending notifications: ' . $e->getMessage());
            return back()->with('error', 'Error resending notifications: ' . $e->getMessage());
        }
    }

    // Download ticket by code
    public function downloadByCode($code)
    {
        $complimentaryCode = ComplimentaryCode::where('code', $code)
            ->where('expires_at', '>', now())
            ->firstOrFail();

        $ticket = new Ticket([
            'reference' => 'TEMP-' . Str::uuid(),
            'event_id' => $complimentaryCode->event_id,
            'first_name' => $complimentaryCode->recipient_name ?? 'Guest',
            'is_complimentary' => true,
            'complimentary_code' => $code
        ]);

        $pdf = $this->generateTicketPdf($ticket);
        return $pdf->download('ticket-' . $code . '.pdf');
    }

    // Download ticket

    public function downloadTicket($ticketId)
    {
        $ticket = ComplimentaryTicket::where('reference', 'LIKE', 'COMP-%')->findOrFail($ticketId);

        $qrCode = QrCode::format('svg')->size(150)->errorCorrection('H')->generate($ticket->reference);

        $pdf = Pdf::loadView('complimentary.ticket-pdf', [
            'ticket' => $ticket,
            'qrCode' => base64_encode($qrCode),
            'hasQrCode' => true
        ]);

        return $pdf->download('complimentary-ticket-' . $ticket->reference . '.pdf');
    }

    // ========== PROTECTED HELPER METHODS ========== //

    protected function validateRequest(Request $request)
    {
        $rules = [
            'event_id' => 'required|exists:events,id',
            'quantity' => 'required|integer|min:1|max:20',
            'sms_template' => 'required|in:simple,detailed',
            'ticket_type' => 'required|in:normal,vip',
            'recipient_name' => 'required|string|max:255',
            'recipient_phone' => 'required|string|max:20',
            'recipient_email' => 'required|email|max:255',
            'generated_by' => 'nullable|string|max:255',
        ];

        return $request->validate($rules);
    }

    protected function prepareRecipients($validatedData)
    {
        $recipients = [];
        if ($validatedData['recipient_type'] === 'bulk') {
            if ($validatedData['bulk_type'] === 'csv') {
                $file = $validatedData['bulk_recipients'];
                $fileContent = file_get_contents($file->getRealPath());
                $lines = array_map('trim', explode("\n", trim($fileContent)));
            } else {
                $lines = array_map('trim', explode("\n", trim($validatedData['manual_recipient'])));
            }
            foreach ($lines as $line) {
                if (empty($line))
                    continue;
                $parts = preg_split('/[,;\t]/', $line);
                $phone = $this->cleanPhoneNumber($parts[0] ?? '');
                $name = $parts[1] ?? 'Guest';
                $email = $parts[2] ?? null;
                if ($phone) {
                    $recipients[] = [
                        'name' => trim($name),
                        'phone' => $phone,
                        'email' => $email ? filter_var(trim($email), FILTER_VALIDATE_EMAIL) : null
                    ];
                }
            }
        } else {
            $recipients[] = [
                'name' => $validatedData['recipient_name'],
                'phone' => $validatedData['recipient_phone'],
                'email' => $validatedData['recipient_email'] ?? null
            ];
        }
        if (empty($recipients)) {
            throw new \Exception('No valid recipients provided');
        }
        return $recipients;
    }



    protected function sendCodeViaSms($code, $template, $event)
    {
        try {
            // Validate phone number first
            $phone = $this->cleanPhoneNumber($code->recipient_phone);
            if (!$phone) {
                throw new \Exception('Invalid phone number format');
            }

            $redeemUrl = route('complimentary.redeem.form', ['code' => $code->code]);
            $downloadUrl = route('complimentary.ticket.by_code', ['code' => $code->code]);

            // $message = $this->processTemplate(
            //     $this->templates[$template]['template'],
            //     $event,
            //     $code,
            //     $code->recipient_name,
            //     $redeemUrl,
            //     $downloadUrl
            // );

            // $message = "Dear {$code->recipient_name}, Welcome to {$event->name}! Complimentary Ticket #: {$code->complimentary_code} Tickets: {$code->quantity} Get your tickets here: {$redeemUrl}";

            $message = "Dear {$code->recipient_name},Welcome to {$event->name}  \nComplimentary Ticket #: {$code->complimentary_code}  \nTickets: {$code->quantity}  \nGet your tickets here:" . url("/get/complimentary-ticket/{$code->reference}");


            return $this->sendSmsViaWasiliana($phone, $message);
        } catch (\Exception $e) {
            Log::error('SMS sending failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }


    private function sendComplimentaryEmail($to, $subject, $complimentaryCode, $event)
    {
        try {
            $emailHtml = view('emails.complimentary-ticket', [
                'complimentaryCode' => $complimentaryCode,
                'event' => $event,
                'downloadUrl' => url("/get/complimentary-ticket/{$complimentaryCode->reference}")
            ])->render();

            // Check email format + DNS
            if (!filter_var($to, FILTER_VALIDATE_EMAIL)) {
                return ['success' => false, 'error' => 'Invalid email format'];
            }

            if (!checkdnsrr(substr(strrchr($to, "@"), 1), "MX")) {
                return ['success' => false, 'error' => 'Email domain cannot receive mail'];
            }

            $client = new Client();

            $response = $client->post('https://pacesetter.co.ke/api/email-sender.php', [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json'
                ],
                'json' => [
                    'to' => $to,
                    'subject' => $subject,
                    'html' => $emailHtml
                ]
            ]);

            $responseData = json_decode($response->getBody(), true);
            Log::info('Email sent successfully', [
                'recipient' => $to,
                'response' => $responseData
            ]);

            return ['success' => true];
        } catch (\Exception $e) {
            Log::error('Email sending failed', [
                'recipient' => $to,
                'error' => $e->getMessage()
            ]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    protected function sendCodeViaEmail($code, $template, $event)
    {
        try {
            if (!filter_var($code->recipient_email, FILTER_VALIDATE_EMAIL)) {
                throw new \Exception("Invalid email address: {$code->recipient_email}");
            }

            $data = [
                'recipientName' => $code->recipient_name,
                'code' => $code->code,
                'eventName' => $event->name,
                'quantity' => $code->ticket_quantity,
                'redeemUrl' => route('complimentary.redeem.form', ['code' => $code->code]),
                'downloadUrl' => route('complimentary.ticket.by_code', ['code' => $code->code]),
            ];

            Mail::to($code->recipient_email)
                ->send(new \App\Mail\ComplimentaryTicketMail(
                    str_replace('{event_name}', $event->name, $this->emailTemplates[$template]['subject']),
                    $data
                ));

            return ['success' => true];

        } catch (\Exception $e) {
            Log::error("Email send failed to {$code->recipient_email}: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    protected function getSuccessMessage($validated, $smsResults, $emailResults)
    {
        $message = count($smsResults) . ' codes generated successfully. ';

        if ($validated['send_sms'] ?? false) {
            $smsSuccess = collect($smsResults)->where('success', true)->count();
            $smsFailed = count($smsResults) - $smsSuccess;
            $message .= "SMS: {$smsSuccess} sent, {$smsFailed} failed. ";
        }

        $emailSuccess = collect($emailResults)->where('success', true)->count();
        $emailFailed = count($emailResults) - $emailSuccess;
        $message .= "Emails: {$emailSuccess} sent, {$emailFailed} failed.";

        return $message;
    }

    protected function generateTicketPdf($ticket)
    {
        $qrCode = QrCode::format('svg')
            ->size(150)
            ->errorCorrection('H')
            ->generate($ticket->reference);

        return Pdf::loadView('complimentary.ticket-pdf', [
            'ticket' => $ticket,
            'qrCode' => $qrCode
        ]);
    }

    protected function processTemplate($template, $event, $code, $name, $redeemUrl, $downloadUrl)
    {
        return str_replace(
            ['{event_name}', '{code}', '{redeem_url}', '{download_url}', '{name}', '{quantity}'],
            [
                $event->name,
                $code->code,
                $redeemUrl,
                $downloadUrl,
                $name,
                $code->ticket_quantity
            ],
            $template
        );
    }
    protected function sendSmsViaWasiliana($phoneNumber, $message)
    {
        try {
            $cleanedPhone = $this->cleanPhoneNumber($phoneNumber);
            if (!$cleanedPhone) {
                throw new \Exception('Invalid phone number format: ' . $phoneNumber);
            }

            $client = new Client([
                'timeout' => 30,
                'verify' => false,
                'http_errors' => false
            ]);

            $apiKey = base64_decode('MlpxU241TEdQZk55VHB2UklDV2lHM3ZXbkppbTZ6VFI2b1hpbXpnMjlJc24zbmlxU3JYTzhmS3JPdjRGWkRTYQ==');

            $response = $client->post('https://api.wasiliana.com/api/v1/send/sms', [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'ApiKey' => $apiKey,
                ],
                'json' => [
                    'from' => 'PACESETTER',
                    'recipients' => [$cleanedPhone],
                    'message' => $message,
                    'type' => 'plain',
                    'dlr' => 'yes',
                    'unicode' => 'no',
                    'skipOtp' => true,
                    'messageClass' => 'normal'
                ]
            ]);

            $responseData = json_decode($response->getBody(), true);
            $statusCode = $response->getStatusCode();

            if ($statusCode !== 200) {
                throw new \Exception($responseData['message'] ?? "API returned status code $statusCode");
            }

            if (($responseData['status'] ?? 'failed') !== 'success') {
                throw new \Exception('Invalid response from SMS gateway: ' . json_encode($responseData));
            }

            return [
                'success' => true,
                'message_id' => $responseData['data'] ?? null,
                'recipient' => $cleanedPhone,
                'message' => $message
            ];
        } catch (\Exception $e) {
            Log::error('SMS sending failed', [
                'error' => $e->getMessage(),
                'phone' => $phoneNumber,
                'message' => $message
            ]);
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    protected function cleanPhoneNumber($phone)
    {
        $phone = trim($phone);
        if (strpos($phone, '+') === 0) {
            $phone = substr($phone, 1);
        }
        if (strpos($phone, '254') === 0 && strlen($phone) === 12) {
            return $phone;
        } elseif (preg_match('/^(07|01)\d{8}$/', $phone)) {
            return '254' . substr($phone, 1);
        } elseif (preg_match('/^7\d{8}$/', $phone)) {
            return '254' . $phone;
        } elseif (preg_match('/^1\d{8}$/', $phone)) {
            return '254' . $phone;
        }
        return false;
    }
    public function __construct()
    {
        // Create separate email log channel
        config([
            'logging.channels.emails' => [
                'driver' => 'single',
                'path' => storage_path('logs/emails.log'),
                'level' => 'debug',
            ]
        ]);
    }

    public function showTicketPreview($ticketId)
    {
        $ticket = ComplimentaryTicket::where('reference', 'LIKE', 'COMP-%')->findOrFail($ticketId);
        return view('complimentary.ticket-preview', [
            'ticket' => $ticket,
        ]);
    }
    public function ticketsIndex(Request $request)
    {
        $query = ComplimentaryTicket::with(['event'])->orderBy('created_at', 'desc');

        // Handle search query
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('reference', 'like', "%{$search}%")
                    ->orWhere('first_name', 'like', "%{$search}%")
                    ->orWhere('last_name', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Handle status filter (based on is_scanned)
        if ($status = $request->input('status')) {
            $query->where('is_scanned', $status === 'scanned' ? 1 : 0);
        }

        // Handle pagination size
        $perPage = $request->input('per_page', 30);
        $perPage = in_array($perPage, [25, 50, 100, 200]) ? $perPage : 30;

        // Paginate results
        $tickets = $query->paginate($perPage)->withQueryString();

        return view('complimentary.tickets-index', compact('tickets'));
    }

}