<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class ContactImportController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:manage-contacts');
    }

    public function showImportForm()
    {
        $totalContacts = DB::table('contacts')->count();
        return view('contacts.import', compact('totalContacts'));
    }

    // ... (rest of your methods remain unchanged)


    public function importCsv(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'csv_file' => 'required|file|mimes:csv,txt|max:2048',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        try {
            $file = $request->file('csv_file');
            $filename = time() . '_contacts.csv';
            $path = $file->storeAs('imports', $filename, 'public');

            // Read and parse CSV
            $csvData = $this->parseCsv(storage_path('app/public/' . $path));

            if (empty($csvData)) {
                return back()->with('error', 'No valid data found in the CSV file. Please ensure your CSV file has at least 4 columns (Name, Phone No., Social, School) in the correct order, and that each row contains a valid name and phone number. Empty rows and rows with missing required fields are automatically skipped. You can download a sample CSV file to see the correct format.');
            }

            // Clear existing contacts if requested
            // if ($request->has('clear_existing')) {
            //     DB::table('contacts')->truncate();
            //     Log::info('Existing contacts cleared before import');
            // }

            // Get existing phone numbers (normalized to last 9 digits) to check for duplicates
            $existingPhoneNumbers = DB::table('contacts')
                ->select(DB::raw('RIGHT(phone_no, 9) as normalized_phone'))
                ->pluck('normalized_phone')
                ->toArray();

            // Filter out contacts that already exist
            $newContacts = [];
            $skippedCount = 0;
            $normalizedBatchPhones = []; // Track normalized phones in current batch

            foreach ($csvData as $contact) {
                // Normalize phone number to last 9 digits for comparison
                $normalizedPhone = substr($contact['phone_no'], -9);

                if (!in_array($normalizedPhone, $existingPhoneNumbers) && !in_array($normalizedPhone, $normalizedBatchPhones)) {
                    $newContacts[] = $contact;
                    $normalizedBatchPhones[] = $normalizedPhone; // Track in current batch to avoid duplicates within CSV
                } else {
                    $skippedCount++;
                    Log::info("Skipping duplicate contact: {$contact['phone_no']} ({$contact['name']}) - normalized: {$normalizedPhone}");
                }
            }

            // Insert data in chunks for better performance
            $chunks = array_chunk($newContacts, 100);
            $totalInserted = 0;

            DB::transaction(function () use ($chunks, &$totalInserted) {
                foreach ($chunks as $chunk) {
                    if (!empty($chunk)) {
                        DB::table('contacts')->insert($chunk);
                        $totalInserted += count($chunk);
                    }
                }
            });

            // Clean up uploaded file
            if (file_exists(storage_path('app/public/' . $path))) {
                unlink(storage_path('app/public/' . $path));
            }

            Log::info("Successfully imported {$totalInserted} contacts. Skipped {$skippedCount} duplicates.");

            $message = "Successfully imported {$totalInserted} contacts!";
            if ($skippedCount > 0) {
                $message .= " ({$skippedCount} duplicate" . ($skippedCount > 1 ? 's' : '') . " skipped)";
            }

            return back()->with('success', $message);
        } catch (\Exception $e) {
            Log::error('CSV Import Error: ' . $e->getMessage());
            return back()->with('error', 'An error occurred during import: ' . $e->getMessage());
        }
    }

    private function parseCsv($filePath)
    {
        $csvData = [];
        $header = null;

        if (($handle = fopen($filePath, 'r')) !== false) {
            $rowNumber = 0;

            while (($row = fgetcsv($handle, 1000, ',')) !== false) {
                $rowNumber++;

                // Skip empty rows
                if (empty(array_filter($row))) {
                    continue;
                }

                // First row is header
                if ($header === null) {
                    $header = array_map('trim', $row);
                    continue;
                }

                // Ensure we have at least the required columns
                if (count($row) >= 4) {
                    $name = trim($row[0]);
                    $phone_no = trim($row[1]);
                    $socials = trim($row[2]);
                    $school_name = trim($row[3]);

                    // Skip if name or phone is empty
                    if (empty($name) || empty($phone_no)) {
                        Log::warning("Skipping row {$rowNumber}: Missing name or phone number");
                        continue;
                    }

                    // Clean phone number (remove spaces, dashes, etc.)
                    $phone_no = preg_replace('/[^0-9]/', '', $phone_no);

                    $csvData[] = [
                        'name' => $name,
                        'phone_no' => $phone_no,
                        'socials' => empty($socials) ? null : $socials,
                        'school_name' => empty($school_name) ? 'Unknown' : $school_name,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                } else {
                    Log::warning("Skipping row {$rowNumber}: Insufficient columns");
                }
            }

            fclose($handle);
        }

        return $csvData;
    }

    public function downloadSample()
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="contacts_sample.csv"',
        ];

        $sampleData = [
            ['Name', 'Phone No.', 'Social', 'School'],
            ['John Doe', '0712345678', 'johndoe123', 'Sample School'],
            ['Jane Smith', '0723456789', 'janesmith456', 'Another School'],
        ];

        $callback = function () use ($sampleData) {
            $file = fopen('php://output', 'w');

            foreach ($sampleData as $row) {
                fputcsv($file, $row);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function storeSingle(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone_no' => 'required|string|max:20',
            'socials' => 'nullable|string|max:255',
            'school_name' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        try {
            // Clean phone number (remove spaces, dashes, etc.)
            $phoneNo = preg_replace('/[^0-9]/', '', $request->phone_no);

            // Normalize phone number to last 9 digits for comparison
            $normalizedPhone = substr($phoneNo, -9);

            // Check if contact already exists (compare using last 9 digits)
            $existingContact = DB::table('contacts')
                ->whereRaw('RIGHT(phone_no, 9) = ?', [$normalizedPhone])
                ->first();

            if ($existingContact) {
                return back()->with('error', 'A contact with this phone number already exists!');
            }

            // Insert new contact
            DB::table('contacts')->insert([
                'name' => trim($request->name),
                'phone_no' => $phoneNo,
                'socials' => $request->socials ? trim($request->socials) : null,
                'school_name' => $request->school_name ? trim($request->school_name) : 'Unknown',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            Log::info("Successfully added single contact: {$request->name} ({$phoneNo})");

            return back()->with('success', "Contact '{$request->name}' has been added successfully!");
        } catch (\Exception $e) {
            Log::error('Single contact creation error: ' . $e->getMessage());
            return back()->with('error', 'An error occurred while adding the contact: ' . $e->getMessage());
        }
    }

    public function clearAll()
    {
        try {
            $count = DB::table('contacts')->count();
            DB::table('contacts')->truncate();

            Log::info("Cleared all {$count} contacts");

            return back()->with('success', "Successfully cleared all {$count} contacts!");
        } catch (\Exception $e) {
            Log::error('Clear contacts error: ' . $e->getMessage());
            return back()->with('error', 'An error occurred while clearing contacts: ' . $e->getMessage());
        }
    }
}
