<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Models\Contact;
use App\Models\SmsLog;
use App\Models\MessageTemplate;
use Illuminate\Support\Facades\Validator;
use GuzzleHttp\Client;

class SmsController extends Controller
{
    public function __construct()
    {
        // Apply permission middleware to all methods (guarded for static analysis)
        $middlewareMethod = 'middleware';
        if (method_exists($this, $middlewareMethod)) {
            $this->{$middlewareMethod}('permission:access-messages')->only([
                'index', 'getAllContactIds', 'compose', 'send', 'logs', 'export'
            ]);
        }
    }

    /**
     * Get current authenticated user or null
     *
     * @return \App\Models\User|null
     */
    private function currentUser()
    {
        return Auth::user();
    }

    /**
     * Get current authenticated user id or null
     */
    private function currentUserId()
    {
        return Auth::id();
    }

    /**
     * Safely check if current user has a given permission.
     * Returns false if there is no user or the Spatie trait isn't present.
     */
    private function userHasPermission(string $permission): bool
    {
        $user = $this->currentUser();
        if (!$user) {
            return false;
        }

        $uses = class_uses(get_class($user)) ?: [];
        if (!in_array(\Spatie\Permission\Traits\HasRoles::class, $uses, true)) {
            return false;
        }

        try {
            if (is_callable([$user, 'can'])) {
                return (bool) call_user_func([$user, 'can'], $permission);
            }

            if (is_callable([$user, 'hasPermissionTo'])) {
                return (bool) call_user_func([$user, 'hasPermissionTo'], $permission);
            }
        } catch (\Throwable $e) {
            Log::warning('Permission check failed: ' . $e->getMessage());
        }

        return false;
    }

    /**
     * Safely check if current user has a given role (or any of an array of roles).
     */
    private function userHasRole(string|array $role): bool
    {
        $user = $this->currentUser();
        if (!$user) {
            return false;
        }

        $uses = class_uses(get_class($user)) ?: [];
        if (!in_array(\Spatie\Permission\Traits\HasRoles::class, $uses, true)) {
            return false;
        }

        try {
            if (is_array($role)) {
                if (is_callable([$user, 'hasAnyRole'])) {
                    return (bool) call_user_func([$user, 'hasAnyRole'], $role);
                }
            } else {
                if (is_callable([$user, 'hasRole'])) {
                    return (bool) call_user_func([$user, 'hasRole'], $role);
                }
            }
        } catch (\Throwable $e) {
            Log::warning('Role check failed: ' . $e->getMessage());
        }

        return false;
    }

    public function index(Request $request)
    {
        // Additional authorization check (redundant but safe)
        if (!$this->currentUser() || !$this->userHasPermission('access-messages')) {
            abort(403, 'Unauthorized access to SMS Manager.');
        }

        $search = $request->get('search');
        $school = $request->get('school');
        $perPage = $request->get('per_page', 50);

        $query = Contact::query();

        // Search functionality
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('phone_no', 'like', "%{$search}%")
                    ->orWhere('socials', 'like', "%{$search}%");
            });
        }

        // School filter
        if ($school) {
            $query->where('school_name', $school);
        }

        $contacts = $query->orderBy('name')
            ->paginate($perPage)
            ->withQueryString();

        // Get all schools for filter dropdown
        $schools = Contact::distinct()
            ->whereNotNull('school_name')
            ->pluck('school_name')
            ->sort()
            ->values();

        // Get SMS stats - hide cost for customer support role
        $smsStats = [
            'total_contacts' => Contact::count(),
            'total_sent' => SmsLog::sent()->count(),
            'total_failed' => SmsLog::failed()->count(),
            'total_pending' => SmsLog::pending()->count(),
            'total_cost' => SmsLog::sent()->sum('cost'),
        ];

        // Hide cost information from customer support
        if ($this->userHasRole('customer-support')) {
            $smsStats['total_cost'] = 0; // Hide actual cost
        }

        return view('sms.index', compact('contacts', 'schools', 'smsStats', 'search', 'school', 'perPage'));
    }

    public function getAllContactIds(Request $request)
    {
        // Authorization check
        if (!$this->currentUser() || !$this->userHasPermission('access-messages')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $search = $request->get('search');
        $school = $request->get('school');

        $query = Contact::query();

        // Apply same filters as index method to get all matching results
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('phone_no', 'like', "%{$search}%")
                    ->orWhere('socials', 'like', "%{$search}%");
            });
        }

        if ($school) {
            $query->where('school_name', $school);
        }

        // Get all contact IDs that match the current filters (across all pages)
        $contactIds = $query->orderBy('name')->pluck('id')->toArray();
        $totalCount = count($contactIds);

        return response()->json([
            'contact_ids' => $contactIds,
            'total_count' => $totalCount
        ]);
    }

    public function compose(Request $request)
    {
        // Authorization check
        if (!$this->currentUser() || !$this->userHasPermission('access-messages')) {
            abort(403, 'Unauthorized access to compose SMS.');
        }

        $selectedIds = $request->get('selected_contacts', []);

        if (empty($selectedIds)) {
            return redirect()->route('sms.index')->with('error', 'Please select at least one contact to send SMS.');
        }

        $contacts = Contact::whereIn('id', $selectedIds)->get();

        // Get active message templates
        $messageTemplates = MessageTemplate::active()->orderBy('name')->get();

        return view('sms.compose', compact('contacts', 'messageTemplates'));
    }

    public function send(Request $request)
    {
        // Authorization check
        if (!$this->currentUser() || !$this->userHasPermission('access-messages')) {
            abort(403, 'Unauthorized access to send SMS.');
        }

        $validator = Validator::make($request->all(), [
            'message' => 'required|string|max:1000',
            'selected_contacts' => 'required|array|min:1',
            'selected_contacts.*' => 'exists:contacts,id',
            'campaign_name' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $contactIds = $request->selected_contacts;
        $message = $request->message;
        $campaignName = $request->campaign_name ?: 'Campaign-' . date('Y-m-d-H-i-s');

        try {
            $contacts = Contact::whereIn('id', $contactIds)->get();
            $successCount = 0;
            $failureCount = 0;
            $totalCost = 0;

            DB::transaction(function () use ($contacts, $message, $campaignName, &$successCount, &$failureCount, &$totalCost) {
                foreach ($contacts as $contact) {
                    // Replace placeholders in message
                    $personalizedMessage = $this->personalizeMessage($message, $contact);

                    // Calculate SMS parts (160 chars per part)
                    $parts = ceil(strlen($personalizedMessage) / 160);

                    // Create SMS log entry
                    $smsLog = SmsLog::create([
                        'contact_name' => $contact->name,
                        'phone_number' => $contact->phone_no,
                        'school_name' => $contact->school_name,
                        'message_content' => $personalizedMessage,
                        'campaign_name' => $campaignName,
                        'parts' => $parts,
                        'status' => 'pending',
                        'sent_by' => $this->currentUserId(), // Track who sent the message
                    ]);

                    // Send SMS via Wassilian API
                    $result = $this->sendSmsViaWasiliana($contact->phone_no, $personalizedMessage);

                    if ($result['success']) {
                        $smsLog->markAsSent($result['message_id'], $result['cost']);
                        $successCount++;
                        $totalCost += $result['cost'];
                    } else {
                        $smsLog->markAsFailed($result['error_data']);
                        $failureCount++;
                    }

                    // Small delay to avoid rate limiting
                    usleep(100000); // 0.1 second delay
                }
            });

            Log::info("SMS Campaign '{$campaignName}' completed by user " . $this->currentUserId() . ". Success: {$successCount}, Failed: {$failureCount}, Cost: {$totalCost}");

            $message = "SMS campaign completed! ";
            $message .= "✅ {$successCount} sent successfully, ";
            if ($failureCount > 0) {
                $message .= "❌ {$failureCount} failed, ";
            }
            
            // Only show cost to authorized roles (not customer support)
            if ($this->userHasPermission('view-metrics')) {
                $message .= "💰 KSH " . number_format($totalCost, 2);
            }

            return redirect()->route('sms.index')->with('success', $message);
        } catch (\Exception $e) {
            Log::error('SMS Campaign Error by user ' . Auth::id() . ': ' . $e->getMessage());
            return back()->with('error', 'An error occurred while sending SMS: ' . $e->getMessage())->withInput();
        }
    }

    public function logs(Request $request)
    {
        // Authorization check
        if (!$this->currentUser() || !$this->userHasPermission('access-messages')) {
            abort(403, 'Unauthorized access to SMS logs.');
        }

        $campaign = $request->get('campaign');
        $status = $request->get('status');
        $perPage = $request->get('per_page', 50);

        $query = SmsLog::query()->orderBy('created_at', 'desc');

        // Customer support can only see their own sent messages
        if ($this->userHasRole('customer-support')) {
            $query->where('sent_by', $this->currentUserId());
        }

        if ($campaign) {
            $query->forCampaign($campaign);
        }

        if ($status) {
            $query->where('status', $status);
        }

        $logs = $query->paginate($perPage)->withQueryString();

        // Get unique campaigns for filter
        $campaigns = SmsLog::distinct()
            ->whereNotNull('campaign_name')
            ->when($this->userHasRole('customer-support'), function ($query) {
                $query->where('sent_by', $this->currentUserId());
            })
            ->pluck('campaign_name')
            ->sort()
            ->values();

        // Get statistics - filter by user for customer support
        $statsQuery = SmsLog::query();
        if ($this->userHasRole('customer-support')) {
            $statsQuery->where('sent_by', $this->currentUserId());
        }

        $stats = [
            'total_messages' => $statsQuery->count(),
            'sent_today' => $statsQuery->clone()->sent()->whereDate('sent_at', today())->count(),
            'total_cost_today' => $statsQuery->clone()->sent()->whereDate('sent_at', today())->sum('cost'),
            'failed_today' => $statsQuery->clone()->failed()->whereDate('created_at', today())->count(),
        ];

        // Hide cost for customer support
        if ($this->userHasRole('customer-support')) {
            $stats['total_cost_today'] = 0;
        }

        return view('sms.logs', compact('logs', 'campaigns', 'stats', 'campaign', 'status', 'perPage'));
    }

    public function export(Request $request)
    {
        // Authorization check
        if (!$this->currentUser() || !$this->userHasPermission('access-messages')) {
            abort(403, 'Unauthorized access to export SMS logs.');
        }

        $campaign = $request->get('campaign');
        $status = $request->get('status');

        $query = SmsLog::query();

        // Customer support can only export their own logs
        if ($this->userHasRole('customer-support')) {
            $query->where('sent_by', $this->currentUserId());
        }

        if ($campaign) {
            $query->forCampaign($campaign);
        }

        if ($status) {
            $query->where('status', $status);
        }

        $logs = $query->orderBy('created_at', 'desc')->get();

        $filename = 'sms_logs_' . date('Y-m-d_H-i-s') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function () use ($logs) {
            $file = fopen('php://output', 'w');

            // CSV headers
            $headers = [
                'ID',
                'Contact Name',
                'Phone Number',
                'School',
                'Message',
                'Status',
                'Parts',
                'Campaign',
                'Sent At',
                'Created At'
            ];

            // Only include cost column for authorized roles
            if ($this->userHasPermission('view-metrics')) {
                array_splice($headers, 7, 0, 'Cost'); // Insert Cost after Parts
            }

            fputcsv($file, $headers);

            foreach ($logs as $log) {
                $row = [
                    $log->id,
                    $log->contact_name,
                    $log->phone_number,
                    $log->school_name,
                    $log->message_content,
                    $log->status,
                    $log->parts,
                    $log->campaign_name,
                    $log->sent_at ? $log->sent_at->format('Y-m-d H:i:s') : '',
                    $log->created_at->format('Y-m-d H:i:s'),
                ];

                // Insert cost if authorized
                if ($this->userHasPermission('view-metrics')) {
                    array_splice($row, 7, 0, $log->cost); // Insert cost after parts
                }

                fputcsv($file, $row);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function personalizeMessage($message, $contact)
    {
        $firstName = explode(' ', $contact->name)[0];

        $replacements = [
            '{first-name}' => $firstName,
            '{name}' => $contact->name,
            '{school}' => $contact->school_name,
        ];

        return str_replace(array_keys($replacements), array_values($replacements), $message);
    }

    private function sendSmsViaWasiliana($phoneNumber, $message)
    {
        try {
            // Clean phone number (ensure it starts with 254)
            // $cleanedPhone = $this->cleanPhoneNumber($phoneNumber);

            // if (!$cleanedPhone) {
            //     return [
            //         'success' => false,
            //         'error_data' => ['error' => 'Invalid phone number format']
            //     ];
            // }

            $cleanedPhone = '254' . substr($phoneNumber, -9);

            // Use the same working configuration from Homecontroller
            $client = new Client();

            $headers = [
                'Content-Type' => 'application/json',
                'ApiKey' => '2ZqSn5LGPfNhtyTpv42RICrWiG3vWnJim6zTR42634rgfvoXimzg29Isn3niqSrXO824fKrOvrwbrew4FZDSa',
            ];

            $body = [
                'from' => 'PACE EVENTS',
                'recipients' => [$cleanedPhone],
                'message' => $message,
            ];

            $response = $client->post('https://api.wasiliana.com/api/v1/send/sms', [
                'headers' => $headers,
                'json' => $body
            ]);

            $responseData = json_decode($response->getBody(), true);

            Log::info('SMS sent successfully via Wasiliana by user ' . Auth::id(), [
                'recipient' => $cleanedPhone,
                'response' => $responseData
            ]);

            // Check if the response indicates success
            if ($response->getStatusCode() === 200) {
                return [
                    'success' => true,
                    'message_id' => $responseData['messageId'] ?? $responseData['id'] ?? null,
                    'cost' => $responseData['cost'] ?? 1.00, // Default cost if not provided
                ];
            } else {
                Log::error('Wasiliana API Error: ' . $response->getBody());
                return [
                    'success' => false,
                    'error_data' => [
                        'status' => $response->getStatusCode(),
                        'response' => $responseData
                    ]
                ];
            }
        } catch (\Exception $e) {
            Log::error('SMS sending error by user ' . Auth::id() . ': ' . $e->getMessage());
            return [
                'success' => false,
                'error_data' => ['error' => $e->getMessage()]
            ];
        }
    }

    private function cleanPhoneNumber($phone)
    {
        // Remove all non-numeric characters
        $cleaned = preg_replace('/[^0-9]/', '', $phone);

        // Handle different formats
        if (strlen($cleaned) === 10 && str_starts_with($cleaned, '07')) {
            // Convert 07XXXXXXXX to 2547XXXXXXXX
            return '254' . substr($cleaned, 1);
        } elseif (strlen($cleaned) === 9 && str_starts_with($cleaned, '7')) {
            // Convert 7XXXXXXXX to 2547XXXXXXXX
            return '254' . $cleaned;
        } elseif (strlen($cleaned) === 12 && str_starts_with($cleaned, '254')) {
            // Already in correct format
            return $cleaned;
        }

        return null; // Invalid format
    }
}